﻿using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Data;
using System.Drawing;
using System.Linq;
using System.Text;
using System.Windows.Forms;
using System.Diagnostics;

using Ivi.Driver.Interop;
using Agilent.AgM9393.Interop;

namespace VSG_SFP
{
    public partial class vsaMonitor_M9393A : Form
    {
        public IAgM9393 Vsa;
        private bool updating;
        private Stopwatch sw1 = new Stopwatch();
        public vsaMonitor_M9393A(IAgM9393 myVsa)
        {
            InitializeComponent();
            Vsa = myVsa;
            updateGuiControls();

            // Init the Spectrum Graph
            spectrumChart.Series.Clear();
            spectrumChart.Series.Add("Spectrum");
            spectrumChart.Series["Spectrum"].ChartType = System.Windows.Forms.DataVisualization.Charting.SeriesChartType.Line;

            // Init the IQ Graph
            iqChart.Series.Clear();
            iqChart.Series.Add("Spectrum");
            iqChart.Series["Spectrum"].ChartType = System.Windows.Forms.DataVisualization.Charting.SeriesChartType.Line;

            // Init the FFT Graph
            fftChart.Series.Clear();
            fftChart.Series.Add("Spectrum");
            fftChart.Series["Spectrum"].ChartType = System.Windows.Forms.DataVisualization.Charting.SeriesChartType.Line;

            switch (Vsa.AcquisitionMode)
            {
                case AgM9393AcquisitionModeEnum.AgM9393AcquisitionModeIQ:
                    tabControl1.SelectedIndex = 2;
                    break;
                case AgM9393AcquisitionModeEnum.AgM9393AcquisitionModePower:
                    tabControl1.SelectedIndex = 0;
                    break;
                case AgM9393AcquisitionModeEnum.AgM9393AcquisitionModeSpectrum:
                    tabControl1.SelectedIndex = 1;
                    break;
                case AgM9393AcquisitionModeEnum.AgM9393AcquisitionModeFFT:
                    tabControl1.SelectedIndex = 3;
                    break;
            }

        }

        private const int CP_NOCLOSE_BUTTON = 0x200;
        protected override CreateParams CreateParams
        {
            get
            {
                CreateParams myCp = base.CreateParams;
                myCp.ClassStyle = myCp.ClassStyle | CP_NOCLOSE_BUTTON;
                return myCp;
            }
        }

        private void UpdateButton_Click(object sender, EventArgs e)
        {
            updateGuiControls();
        }

        private void updateGuiControls()
        {
            updating = true;
            // Update RF Controls
            FreqMHzControl.Value = (decimal)(Vsa.RF.Frequency / 1e6);
            RefLevel_dBmControl.Value = (decimal)Vsa.RF.Power;
            Vsa.Modules.Downconverter.Attenuator.ActiveDownconverterAttenuator = "Atten3";
            digLevelOffset.Value = (decimal)Vsa.Modules.Downconverter.Attenuator.Offset;
            ifBwMhzControl.Value = (decimal)(Vsa.RF.IFBandwidth / 1e6);
            // Read Conversion Mode and set Control to correct value from enum
            AgM9393ConversionEnum convMode = Vsa.RF.Conversion;
            switch (convMode)
            {
                case AgM9393ConversionEnum.AgM9393ConversionAuto:
                    conversionModeBox.SelectedIndex = 3;
                    break;
                case AgM9393ConversionEnum.AgM9393ConversionImageProtect:
                    conversionModeBox.SelectedIndex = 2;
                    break;
                case AgM9393ConversionEnum.AgM9393ConversionSingleHighSide:
                    conversionModeBox.SelectedIndex = 0;
                    break;
                case AgM9393ConversionEnum.AgM9393ConversionSingleLowSide:
                    conversionModeBox.SelectedIndex = 1;
                    break;
            }
            AgM9393AcquisitionModeEnum acqMode = Vsa.AcquisitionMode;
            switch (acqMode)
            {
                case AgM9393AcquisitionModeEnum.AgM9393AcquisitionModeIQ:
                    acqModeBox.SelectedIndex = 0;
                    break;
                case AgM9393AcquisitionModeEnum.AgM9393AcquisitionModeSpectrum:
                    acqModeBox.SelectedIndex = 1;
                    break;
                case AgM9393AcquisitionModeEnum.AgM9393AcquisitionModePower:
                    acqModeBox.SelectedIndex = 2;
                    break;
                case AgM9393AcquisitionModeEnum.AgM9393AcquisitionModeFFT:
                    acqModeBox.SelectedIndex = 3;
                    break;
            }

            // Trigger Controls
            AgM9393AcquisitionTriggerModeEnum trigMode = Vsa.Triggers.AcquisitionTrigger.Mode;
            switch (trigMode)
            {
                case AgM9393AcquisitionTriggerModeEnum.AgM9393AcquisitionTriggerModeImmediate:
                    immTrigButton.Checked = true;
                    break;
                case AgM9393AcquisitionTriggerModeEnum.AgM9393AcquisitionTriggerModeExternal:
                    extTrigButton.Checked = true;
                    break;
            }
            triggerDelay_us.Value = (decimal) (Vsa.Triggers.AcquisitionTrigger.Delay * 1e6);

            // Update Power Acquisition Mode Controls
            bwControl.Value = (decimal)(Vsa.PowerAcquisition.Bandwidth / 1e6);
            durationControl.Value = (decimal)(Vsa.PowerAcquisition.Duration * 1e6);
            pwrOffsetFreqControl.Value = (decimal)(Vsa.PowerAcquisition.OffsetFrequency / 1e6);
            filterAlphaControl.Value = (decimal)Vsa.PowerAcquisition.ChannelFilter.Alpha;
            filterBwControl.Value = (decimal)(Vsa.PowerAcquisition.ChannelFilter.Bandwidth / 1e6);
            AgM9393ChannelFilterShapeEnum chanFilt = Vsa.PowerAcquisition.ChannelFilter.Shape;
            switch (chanFilt)
            {
                case AgM9393ChannelFilterShapeEnum.AgM9393ChannelFilterShapeGaussian:
                    filterShapeControl.SelectedIndex = 2;
                    break;
                case AgM9393ChannelFilterShapeEnum.AgM9393ChannelFilterShapeNone:
                    filterShapeControl.SelectedIndex = 0;
                    break;
                case AgM9393ChannelFilterShapeEnum.AgM9393ChannelFilterShapeRaisedCosine:
                    filterShapeControl.SelectedIndex = 3;
                    break;
                case AgM9393ChannelFilterShapeEnum.AgM9393ChannelFilterShapeRectangular:
                    filterShapeControl.SelectedIndex = 1;
                    break;
                case AgM9393ChannelFilterShapeEnum.AgM9393ChannelFilterShapeRootRaisedCosine:
                    filterShapeControl.SelectedIndex = 4;
                    break;
            }

            // Update Spectrum Acqusition Controls
            spanMhzControl.Value = (decimal)(Vsa.SpectrumAcquisition.Span / 1e6);
            rbwkHzControl.Value = (decimal)(Vsa.SpectrumAcquisition.ResolutionBandwidth / 1e3);
            AgM9393FFTWindowShapeEnum specWind = Vsa.SpectrumAcquisition.FFTWindowShape;
            switch (specWind)
            {
                case AgM9393FFTWindowShapeEnum.AgM9393FFTWindowShapeUniform:
                    windowTypeBox.SelectedIndex = 0;
                    break;
                case AgM9393FFTWindowShapeEnum.AgM9393FFTWindowShapeHann:
                    windowTypeBox.SelectedIndex = 1;
                    break;
                case AgM9393FFTWindowShapeEnum.AgM9393FFTWindowShapeFlatTop:
                    windowTypeBox.SelectedIndex = 2;
                    break;
                case AgM9393FFTWindowShapeEnum.AgM9393FFTWindowShapeHDRFlatTop:
                    windowTypeBox.SelectedIndex = 3;
                    break;
                case AgM9393FFTWindowShapeEnum.AgM9393FFTWindowShapeGaussian:
                    windowTypeBox.SelectedIndex = 4;
                    break;
            }
            AgM9393SpectrumAveragingEnum specAvg = Vsa.SpectrumAcquisition.Averaging.Mode;
            switch (specAvg)
            {
                case AgM9393SpectrumAveragingEnum.AgM9393SpectrumAveragingOff:
                    avgTypeBox.SelectedIndex = 0;
                    break;
                case AgM9393SpectrumAveragingEnum.AgM9393SpectrumAveragingCounted:
                    avgTypeBox.SelectedIndex = 1;
                    break;
                case AgM9393SpectrumAveragingEnum.AgM9393SpectrumAveragingCountedPeak:
                    avgTypeBox.SelectedIndex = 2;
                    break;
                case AgM9393SpectrumAveragingEnum.AgM9393SpectrumAveragingTime:
                    avgTypeBox.SelectedIndex = 3;
                    break;
                case AgM9393SpectrumAveragingEnum.AgM9393SpectrumAveragingTimePeak:
                    avgTypeBox.SelectedIndex = 4;
                    break;
            }
            avgCountControl.Value = (decimal)Vsa.SpectrumAcquisition.Averaging.Count;
            avgTimemsControl.Value = (decimal)(Vsa.SpectrumAcquisition.Averaging.Duration * 1e3);
            overlapControl.Value = (decimal)Vsa.SpectrumAcquisition.Averaging.Overlap;

            // Update IQ Acquisition Mode Controls
            srControl.Value = (decimal)(Vsa.IQAcquisition.SampleRate / 1e6);
            pointsControl.Value = (decimal)(Vsa.IQAcquisition.Samples);
            iqFilterAlphaControl.Value = (decimal)Vsa.IQAcquisition.ChannelFilter.Alpha;
            iqFilterBwControl.Value = (decimal)(Vsa.IQAcquisition.ChannelFilter.Bandwidth / 1e6);
            AgM9393ChannelFilterShapeEnum iqChanFilt = Vsa.IQAcquisition.ChannelFilter.Shape;
            switch (iqChanFilt)
            {
                case AgM9393ChannelFilterShapeEnum.AgM9393ChannelFilterShapeGaussian:
                    iqFilterShapeControl.SelectedIndex = 2;
                    break;
                case AgM9393ChannelFilterShapeEnum.AgM9393ChannelFilterShapeNone:
                    iqFilterShapeControl.SelectedIndex = 0;
                    break;
                case AgM9393ChannelFilterShapeEnum.AgM9393ChannelFilterShapeRaisedCosine:
                    iqFilterShapeControl.SelectedIndex = 3;
                    break;
                case AgM9393ChannelFilterShapeEnum.AgM9393ChannelFilterShapeRectangular:
                    iqFilterShapeControl.SelectedIndex = 1;
                    break;
                case AgM9393ChannelFilterShapeEnum.AgM9393ChannelFilterShapeRootRaisedCosine:
                    iqFilterShapeControl.SelectedIndex = 4;
                    break;
            }

            // Update FFT Acquisition Mode Controls
            fftSampleRateMHz.Value = (decimal)(Vsa.FFTAcquisition.SampleRate / 1e6);
            fftAverages.Value = (decimal)Vsa.FFTAcquisition.AveragingCount;
            AgM9393FFTWindowShapeEnum fftWind = Vsa.FFTAcquisition.WindowShape;
            switch (fftWind)
            {
                case AgM9393FFTWindowShapeEnum.AgM9393FFTWindowShapeUniform:
                    fftWindowBox.SelectedIndex = 0;
                    break;
                case AgM9393FFTWindowShapeEnum.AgM9393FFTWindowShapeHann:
                    fftWindowBox.SelectedIndex = 1;
                    break;
                case AgM9393FFTWindowShapeEnum.AgM9393FFTWindowShapeFlatTop:
                    fftWindowBox.SelectedIndex = 2;
                    break;
                case AgM9393FFTWindowShapeEnum.AgM9393FFTWindowShapeGaussian:
                    fftWindowBox.SelectedIndex = 4;
                    break;
            }

            // Disable the HDR Flattop window in the list

            switch (Vsa.FFTAcquisition.Samples)
            {
                case 32:
                    fftSizeBox.SelectedIndex = 0;
                    break;
                case 64:
                    fftSizeBox.SelectedIndex = 1;
                    break;
                case 128:
                    fftSizeBox.SelectedIndex = 2;
                    break;
                case 256:
                    fftSizeBox.SelectedIndex = 3;
                    break;
                case 512:
                    fftSizeBox.SelectedIndex = 4;
                    break;
            }

            Application.DoEvents();
            updating = false;
        }

        private void readGuiControls(object sender, EventArgs e)
        {
            if (!updating)
            {
                try
                {
                    // RF Controls
                    Vsa.RF.Frequency = (double)FreqMHzControl.Value * 1e6;
                    Vsa.RF.Power = (double)RefLevel_dBmControl.Value;
                    Vsa.Modules.Downconverter.Attenuator.ActiveDownconverterAttenuator = "Atten3";
                    Vsa.Modules.Downconverter.Attenuator.Offset = (double)digLevelOffset.Value;
                    Vsa.RF.IFBandwidth = (double)ifBwMhzControl.Value * 1e6;
                    Vsa.RF.Conversion = (AgM9393ConversionEnum)(Math.Max(0, conversionModeBox.SelectedIndex));
                    Vsa.AcquisitionMode = (AgM9393AcquisitionModeEnum)(Math.Max(0, acqModeBox.SelectedIndex));

                    switch (Vsa.AcquisitionMode)
                    {
                        case AgM9393AcquisitionModeEnum.AgM9393AcquisitionModeIQ:
                            tabControl1.SelectedIndex = 2;
                            break;
                        case AgM9393AcquisitionModeEnum.AgM9393AcquisitionModePower:
                            tabControl1.SelectedIndex = 0;
                            break;
                        case AgM9393AcquisitionModeEnum.AgM9393AcquisitionModeSpectrum:
                            tabControl1.SelectedIndex = 1;
                            break;
                        case AgM9393AcquisitionModeEnum.AgM9393AcquisitionModeFFT:
                            tabControl1.SelectedIndex = 3;
                            break;
                    }

                    // Trigger Mode
                    if (extTrigButton.Checked)
                    {
                        Vsa.Triggers.AcquisitionTrigger.Mode = AgM9393AcquisitionTriggerModeEnum.AgM9393AcquisitionTriggerModeExternal;
                        Vsa.Triggers.AcquisitionTrigger.Delay = (double)triggerDelay_us.Value / 1e6;
                    }
                    else
                        Vsa.Triggers.AcquisitionTrigger.Mode = AgM9393AcquisitionTriggerModeEnum.AgM9393AcquisitionTriggerModeImmediate;

                    // Power Acquisition Mode Controls
                    Vsa.PowerAcquisition.Bandwidth = (double)bwControl.Value * 1e6;
                    Vsa.PowerAcquisition.Duration = (double)durationControl.Value / 1e6;
                    Vsa.PowerAcquisition.OffsetFrequency = (double)pwrOffsetFreqControl.Value * 1e6;
                    Vsa.PowerAcquisition.ChannelFilter.Alpha = (double)filterAlphaControl.Value;
                    Vsa.PowerAcquisition.ChannelFilter.Bandwidth = (double)filterBwControl.Value * 1e6;
                    Vsa.PowerAcquisition.ChannelFilter.Shape = (AgM9393ChannelFilterShapeEnum)(Math.Max(0, filterShapeControl.SelectedIndex));

                    // Spectrum Acquisition Mode Controls
                    Vsa.SpectrumAcquisition.Span = (double)spanMhzControl.Value * 1e6;
                    Vsa.SpectrumAcquisition.ResolutionBandwidth = (double)rbwkHzControl.Value * 1e3;
                    Vsa.SpectrumAcquisition.FFTWindowShape = (AgM9393FFTWindowShapeEnum)(Math.Max(0, windowTypeBox.SelectedIndex));
                    Vsa.SpectrumAcquisition.Averaging.Mode = (AgM9393SpectrumAveragingEnum)(Math.Max(0, avgTypeBox.SelectedIndex));
                    Vsa.SpectrumAcquisition.Averaging.Count = (int)avgCountControl.Value;
                    Vsa.SpectrumAcquisition.Averaging.Duration = (double)avgTimemsControl.Value / 1e3;
                    Vsa.SpectrumAcquisition.Averaging.Overlap = (double)overlapControl.Value;

                    // IQ Acquisition Mode Controls
                    Vsa.IQAcquisition.SampleRate = (double)srControl.Value * 1e6;
                    Vsa.IQAcquisition.Samples = (int)pointsControl.Value;
                    Vsa.IQAcquisition.ChannelFilter.Alpha = (double)iqFilterAlphaControl.Value;
                    Vsa.IQAcquisition.ChannelFilter.Bandwidth = (double)iqFilterBwControl.Value * 1e6;
                    Vsa.IQAcquisition.ChannelFilter.Shape = (AgM9393ChannelFilterShapeEnum)(Math.Max(0, iqFilterShapeControl.SelectedIndex));

                    // FFT Acquisition Mode Controls
                    Vsa.FFTAcquisition.Configure((AgM9393FFTAcquisitionLengthEnum)Convert.ToInt32(fftSizeBox.SelectedItem.ToString()),
                        (double)fftSampleRateMHz.Value * 1e6,
                        (int)fftAverages.Value,
                        (AgM9393FFTWindowShapeEnum)(Math.Max(0, fftWindowBox.SelectedIndex)),
                        (double)fftOffsetMHz.Value * 1e6);
                    
                    switch (Vsa.AcquisitionMode)
                    {
                        case AgM9393AcquisitionModeEnum.AgM9393AcquisitionModeFFT:
                            Vsa.MemoryMode = AgM9393MemoryModeEnum.AgM9393MemoryModeLargeAcquisition;
                            break;
                        default:
                            Vsa.MemoryMode = AgM9393MemoryModeEnum.AgM9393MemoryModeNormal;
                            break;
                    }

                    Vsa.Apply();
                    if (autoMeasureBox.Checked)
                    {
                        switch (Vsa.AcquisitionMode)
                        {
                            case AgM9393AcquisitionModeEnum.AgM9393AcquisitionModeIQ:
                                measIq_Click(sender, e);
                                break;
                            case AgM9393AcquisitionModeEnum.AgM9393AcquisitionModePower:
                                measPwrButton_Click(sender, e);
                                break;
                            case AgM9393AcquisitionModeEnum.AgM9393AcquisitionModeSpectrum:
                                measSpectrum_Click(sender, e);
                                break;
                            case AgM9393AcquisitionModeEnum.AgM9393AcquisitionModeFFT:
                                fftMeas_Click(sender, e);
                                break;
                        }
                    }
                }

                catch (Exception ex)
                {
                    MessageBox.Show(ex.Message);
                }
            }
        }

        private void vsaMonitor_Load(object sender, EventArgs e)
        {

        }

        private void measPwrButton_Click(object sender, EventArgs e)
        {
            if (Vsa.AcquisitionMode != AgM9393AcquisitionModeEnum.AgM9393AcquisitionModePower)
            {
                MessageBox.Show("M9391A not in Power Acquisition Mode");
                return;
            }
            try
            {
                //Start Test timer
                sw1.Reset();
                sw1.Start();
                bool overload = false;
                double measPower = -999.0;
                Vsa.Arm();
                Vsa.PowerAcquisition.ReadPower(0, AgM9393PowerUnitsEnum.AgM9393PowerUnitsdBm, 
                    ref measPower, ref overload);
                overloadButton.Visible = overload;
                measPowerBox.Text = measPower.ToString("F3");
                pwrMeasTimeBox.Text = sw1.ElapsedMilliseconds.ToString();
                Application.DoEvents();
            }
            catch (Exception ex)
            {
                MessageBox.Show(ex.Message);
            }
        }

        private void measSpectrum_Click(object sender, EventArgs e)
        {
            if (Vsa.AcquisitionMode != AgM9393AcquisitionModeEnum.AgM9393AcquisitionModeSpectrum)
            {
                MessageBox.Show("M9391A not in Spectrum Acquisition Mode");
                return;
            }
            try
            {
                //Start Test timer
                sw1.Reset();
                sw1.Start();
                double fStart = 0;
                double fDelta = 0;
                double[] spectrumData = new double[Vsa.SpectrumAcquisition.Bins];
                bool overload = false;
                Vsa.Arm();
                Vsa.SpectrumAcquisition.ReadPowerSpectrum(0, AgM9393PowerUnitsEnum.AgM9393PowerUnitsdBm,
                    ref spectrumData, ref overload, ref fStart, ref fDelta);

                specMeasTimeBox.Text = sw1.ElapsedMilliseconds.ToString();
                specPeak.Text = spectrumData.Max().ToString("0.00");
                pointsBox.Text = spectrumData.Length.ToString();
                fDeltaBox.Text = (fDelta / 1e3).ToString("0.000");
                spectrumChart.Series["Spectrum"].Points.Clear();
                specOverloadButton.Visible = overload;
                for (int j = 0; j < spectrumData.Length; j++)
                {
                    spectrumChart.Series["Spectrum"].Points.AddXY(((double)j * fDelta + fStart) / 1e6, spectrumData[j]);
                }
                spectrumChart.ChartAreas[0].AxisX.MajorGrid.Interval = spectrumData.Length *fDelta / 10e6;
                //spectrumChart.ChartAreas[0].AxisX.RoundAxisValues();
                //spectrumChart.ChartAreas[0].AxisX.Minimum = fStart / 1e6;
                //spectrumChart.ChartAreas[0].AxisX.Maximum = (fStart + spectrumData.Length) / 1e6;
            }
            catch (Exception ex)
            {
                MessageBox.Show(ex.Message);
            }
        }

        private void measIq_Click(object sender, EventArgs e)
        {
            if (Vsa.AcquisitionMode != AgM9393AcquisitionModeEnum.AgM9393AcquisitionModeIQ)
            {
                MessageBox.Show("M9391A not in IQ Acquisition Mode");
                return;
            }
            try
            {
                //Start Test timer
                sw1.Reset();
                sw1.Start();
                int numSamples = Vsa.IQAcquisition.Samples;
                double sampleTime = 1 / Vsa.IQAcquisition.SampleRate;
                double[] iqData = new double[numSamples * 2];
                double[] powerData = new double[numSamples];
                bool overload = false;
                Vsa.Arm();
                    Vsa.IQAcquisition.ReadIQData(0, AgM9393IQUnitsEnum.AgM9393IQUnitsSquareRootMilliWatts, 0, numSamples, ref iqData, ref overload);
                    iqMeasTimeBox.Text = sw1.ElapsedMilliseconds.ToString();
                    iqChart.Series["Spectrum"].Points.Clear();
                    for (int i = 0; i < numSamples; i++)
                    {
                        powerData[i] = 10 * Math.Log10(iqData[2 * i] * iqData[2 * i] + iqData[(2 * i) + 1] * iqData[(2 * i) + 1]);
                        iqChart.Series["Spectrum"].Points.AddXY(i * sampleTime * 1e6, powerData[i]);
                    }
                iqOverloadButton.Visible = overload;

            }
            catch (Exception ex)
            {
                MessageBox.Show(ex.Message);
            }
        }

        private void fftMeas_Click(object sender, EventArgs e)
        {
            if (Vsa.AcquisitionMode != AgM9393AcquisitionModeEnum.AgM9393AcquisitionModeFFT)
            {
                MessageBox.Show("M9391A not in FFT Acquisition Mode");
                return;
            }
            try
            {
                //Start Test timer
                sw1.Reset();
                sw1.Start();
                int numSamples = Vsa.FFTAcquisition.Samples;
                double sampleTime = 1 / Vsa.FFTAcquisition.SampleRate;
                double[] fftData = new double[numSamples];
                bool overload = false;
                if (fftApply.Checked)
                    Vsa.Apply();
                Vsa.Arm();
                Vsa.WaitForData(1000);
                Vsa.FFTAcquisition.ReadMagnitudeData(0, ref fftData, ref overload);
                fftTime.Text = sw1.ElapsedMilliseconds.ToString();
                fftChart.Series["Spectrum"].Points.Clear();
                double fDelta = Vsa.FFTAcquisition.SampleRate / numSamples;
                double fStart = Vsa.RF.Frequency - (numSamples / 2 * fDelta);
                double fCenter = Vsa.RF.Frequency;
                double fftMax = -200;
                // Don't plot the first and last 10% of data
                for (int i = (int)(numSamples * 0.1) + 1; i < (int)(numSamples * 0.9) - 1; i++)
                {
                    fftChart.Series["Spectrum"].Points.AddXY((fStart + (i * fDelta)) / 1e6, 10 * Math.Log10(Math.Max(fftData[i], 0.5e-13)));
                    // find the maximum of the displayed points
                    if (fftData[i] > fftMax) fftMax = fftData[i];
                }
                fftOverload.Visible = overload;
                double measPout = 0;
                bool isRrcFilter = false;
                bool isGausFilter = false;
                double FilterBw = Vsa.PowerAcquisition.ChannelFilter.Bandwidth;
                double channelBw = FilterBw;
                if (Vsa.PowerAcquisition.ChannelFilter.Shape == AgM9393ChannelFilterShapeEnum.AgM9393ChannelFilterShapeRootRaisedCosine)
                {
                    isRrcFilter = true;
                    // use a wider BW for RRC filter to get the power in the filter skirt
                    channelBw = FilterBw * 1.3;
                }
                if (Vsa.PowerAcquisition.ChannelFilter.Shape == AgM9393ChannelFilterShapeEnum.AgM9393ChannelFilterShapeGaussian)
                {
                    isGausFilter = true;
                    // use a wider BW for Gaussian filter to get the power in the filter skirt
                    channelBw = FilterBw * 2.0;
                }

                // calculate start and stop bin for Reference power, including any fractional bin
                // Try to use the channel power settings to calculate integrated power.  If it does not work, show "---"
                try
                {
                    double fBinStart = fCenter - channelBw / 2;
                    int startBin = (int)Math.Floor((fBinStart - fStart) / fDelta);
                    int stopBin = (int)Math.Ceiling((fBinStart - fStart + channelBw) / fDelta);
                    for (int j = startBin; j <= stopBin; j++)
                    {
                        if (isRrcFilter)
                        {
                            double binFreq = (fStart + j * fDelta) - fCenter;
                            double binScale = getRrcFilterWeightSquared(binFreq, FilterBw, Vsa.PowerAcquisition.ChannelFilter.Alpha);
                            measPout += fftData[j] * binScale;
                        }
                        else if (isGausFilter)
                        {
                            double binFreq = (fStart + j * fDelta) - fCenter;
                            double binScale = getGausFilterWeight(binFreq, FilterBw);
                            measPout += fftData[j] * binScale;
                        }
                        else
                        {
                            // for RECT filtered channels, scale by percentage of bin inside channel for first and last bin
                            if (j == startBin)
                            {
                                double binScale = ((fBinStart - fStart) / fDelta - startBin);
                                measPout += (fftData[j] * binScale);
                            }
                            else if (j == stopBin)
                            {
                                double binScale = (stopBin - (fBinStart - fStart + channelBw) / fDelta);
                                measPout += (fftData[j] * binScale);
                            }
                            else
                                measPout += fftData[j];
                        }

                    }
                    measPout = 10 * Math.Log10(measPout);
                    // Set the Effective Noise Bandwidth based on Window Type
                    // 1.0 for Uniform, 1.5 for Hann, 2.215 for GaussTop, 3.8194 for FlatTop 
                    double windowEnbw = 1.0;
                    switch (Vsa.FFTAcquisition.WindowShape)
                    {
                        case AgM9393FFTWindowShapeEnum.AgM9393FFTWindowShapeUniform:
                            {
                                windowEnbw = 1.0;
                                break;
                            }
                        case AgM9393FFTWindowShapeEnum.AgM9393FFTWindowShapeHann:
                            {
                                windowEnbw = 1.5;
                                break;
                            }
                        case AgM9393FFTWindowShapeEnum.AgM9393FFTWindowShapeGaussian:
                            {
                                windowEnbw = 2.215;
                                break;
                            }
                        case AgM9393FFTWindowShapeEnum.AgM9393FFTWindowShapeFlatTop:
                            {
                                windowEnbw = 3.8194;
                                break;
                            }
                        default:
                            {
                                throw (new Exception("Window Mode Not Supported:" + Vsa.FFTAcquisition.WindowShape.ToString()));
                            }
                    }
                    // Correct for the ENBW of FFT Window choice
                    measPout -= 10 * Math.Log10(windowEnbw);
                    fftPowerBox.Text = measPout.ToString("F3");

                    fftMax = 10 * Math.Log10(fftMax);
                    fftMax -= 10 * Math.Log10(windowEnbw);
                    fftPeak.Text = fftMax.ToString("0.00");
                }
                catch
                {
                    fftPowerBox.Text = "---";
                }
            }
            catch (Exception ex)
            {
                MessageBox.Show(ex.Message);
            }

        }
        public double getRrcFilterWeightSquared(double freq, double symbolRate, double alpha)
        {
            double m_symbolRate;
            double m_alpha;
            double m_lowerBreakPoint_Hz;
            double m_upperBreakPoint_Hz;
            double m_constant1;         // for calculation efficiency only
            double m_constant2;         // for calculation efficiency only

            m_symbolRate = symbolRate;
            m_alpha = alpha;
            m_lowerBreakPoint_Hz = ((1 - alpha) * symbolRate / 2);
            m_upperBreakPoint_Hz = ((1 + alpha) * symbolRate / 2);

            double f = Math.Abs(freq);

            double result;

            if (f <= m_lowerBreakPoint_Hz)
                result = 1;
            else if (f <= m_upperBreakPoint_Hz)
            {
                m_constant1 = (Math.PI / (2 * symbolRate * alpha));
                m_constant2 = (symbolRate * (1 - alpha) / 2);
                result = Math.Cos(m_constant1 * (f - m_constant2));
            }
            else
                result = 0;

            return result * result;
        }

        public double getGausFilterWeight(double freq, double bandwidth)
        {
            double B = 0.5 * bandwidth / (Math.Sqrt(Math.Log(2) / (2 * Math.PI)));
            double f = Math.Abs(freq);
            double result = Math.Exp(-1 * Math.PI * (f / B) * (f / B));

            return result;
        }
    }
}
